/**
 * Shared utilities and constants for EAC Core functionality
 * This file consolidates common functionality used across CoreTests.js and EacTests.js
 */

// =====================
// Shared Constants
// =====================

/**
 * Shared constants for question types and processing
 */
const QUESTION_TYPES = {
	MULTIPLE_CHOICE: 1,
	TRUE_FALSE: 2,
	MULTIPLE_RESPONSE: 3,
	ORDERING: 4,
	MATCHING: 5,
	ESSAY: 6,
	DEMOGRAPHIC: 7,
	FILL_BLANK: 8,
	CALCULATED_VARIABLE: 9,
	SHORT_ANSWER: 10,
	HOT_SPOT: 11,
	JUMBLED_SENTENCE: 12,
	JUMBLED_SENTENCE_ULTRA: 13,
	NUMERIC: 14,
	LIKERT: 15,
	MULTIPLE_CHOICE_ULTRA: 16,
	TRUE_FALSE_ULTRA: 17
};

/**
 * Question types that are not scoreable
 */
const NON_SCOREABLE_TYPES = [
	QUESTION_TYPES.ESSAY,
	QUESTION_TYPES.DEMOGRAPHIC,
	QUESTION_TYPES.SHORT_ANSWER
];

/**
 * Ultra question types
 */
const ULTRA_QUESTION_TYPES = [
	QUESTION_TYPES.MULTIPLE_CHOICE_ULTRA,
	QUESTION_TYPES.TRUE_FALSE_ULTRA,
	QUESTION_TYPES.JUMBLED_SENTENCE_ULTRA
];

// =====================
// Shared Utility Classes
// =====================

/**
 * Core utility functions
 */
class CoreUtils {
	/**
	 * Safe XML text content extraction
	 */
	static getTextContentSafe(parent, tag) {
		if (!parent || !tag) return '';
		const element = parent.querySelector(tag);
		return element ? element.textContent || '' : '';
	}

	/**
	 * Safe XML attribute extraction
	 */
	static getAttributeSafe(el, attr) {
		if (!el || !attr) return '';
		return el.getAttribute(attr) || '';
	}

	/**
	 * Parse score string in legacy format
	 */
	static parseScoreString(str) {
		if (!str || !str.startsWith('s_')) return { score: 0, max: 0 };
		
		const parts = str.substring(2).split('_');
		const score = parseFloat(parts[0]) || 0;
		const max = parseFloat(parts[1]) || 0;
		
		return { score, max };
	}

	/**
	 * Serialize score to legacy string format
	 */
	static serializeScoreString(score, max) {
		return `s_${score}_${max}`;
	}

	/**
	 * Split string by sections (***)
	 */
	static splitSections(str) {
		return str ? str.split('***') : [];
	}

	/**
	 * Split string by fields (||)
	 */
	static splitFields(str) {
		return str ? str.split('||') : [];
	}

	/**
	 * Check if question type is not scoreable
	 */
	static isNotScoreable(kindId) {
		return NON_SCOREABLE_TYPES.includes(parseInt(kindId));
	}

	/**
	 * Check if question type is ultra
	 */
	static isUltraQuestion(kindId) {
		return ULTRA_QUESTION_TYPES.includes(parseInt(kindId));
	}

	/**
	 * Get question type abbreviation
	 */
	static getQuestionTypeAbbreviation(kindId, ultraStatus = 'C') {
		const typeMap = {
			[QUESTION_TYPES.MULTIPLE_CHOICE]: 'MC',
			[QUESTION_TYPES.TRUE_FALSE]: 'TF',
			[QUESTION_TYPES.MULTIPLE_RESPONSE]: 'MR',
			[QUESTION_TYPES.ORDERING]: 'ORD',
			[QUESTION_TYPES.MATCHING]: 'MAT',
			[QUESTION_TYPES.ESSAY]: 'ES',
			[QUESTION_TYPES.DEMOGRAPHIC]: 'DEM',
			[QUESTION_TYPES.FILL_BLANK]: 'FB',
			[QUESTION_TYPES.CALCULATED_VARIABLE]: 'CV',
			[QUESTION_TYPES.SHORT_ANSWER]: 'SA',
			[QUESTION_TYPES.HOT_SPOT]: 'HS',
			[QUESTION_TYPES.JUMBLED_SENTENCE]: 'JS',
			[QUESTION_TYPES.JUMBLED_SENTENCE_ULTRA]: 'JSU',
			[QUESTION_TYPES.NUMERIC]: 'NUM',
			[QUESTION_TYPES.LIKERT]: 'LIK',
			[QUESTION_TYPES.MULTIPLE_CHOICE_ULTRA]: 'MCU',
			[QUESTION_TYPES.TRUE_FALSE_ULTRA]: 'TFU'
		};
		
		return typeMap[kindId] || 'UNK';
	}

	/**
	 * Safe number parsing with fallback
	 */
	static parseNumber(value, fallback = 0) {
		const parsed = parseFloat(value);
		return isNaN(parsed) ? fallback : parsed;
	}

	/**
	 * Safe integer parsing with fallback
	 */
	static parseIntSafe(value, fallback = 0) {
		const parsed = parseInt(value);
		return isNaN(parsed) ? fallback : parsed;
	}

	/**
	 * Get property safely with fallback
	 */
	static getProperty(obj, prop, fallback = '') {
		return obj && obj[prop] !== undefined ? obj[prop] : fallback;
	}

	/**
	 * Format date range for display
	 */
	static formatDateRange(minDate, maxDate, lang = 'en_US') {
		if (!minDate || !maxDate) return '';
		
		const formatDate = (date) => {
			if (typeof date === 'string') {
				date = new Date(date);
			}
			return date.toLocaleDateString(lang === 'en_US' ? 'en-US' : 'en-GB');
		};
		
		return `${formatDate(minDate)} - ${formatDate(maxDate)}`;
	}

	/**
	 * Calculate date statistics from raw scores
	 */
	static calculateDateStats(rawScores) {
		if (!rawScores || rawScores.length === 0) {
			return { minDate: null, maxDate: null, dateRange: '' };
		}

		const dates = rawScores
			.map(score => new Date(score.mdate))
			.filter(date => !isNaN(date.getTime()))
			.sort((a, b) => a - b);

		if (dates.length === 0) {
			return { minDate: null, maxDate: null, dateRange: '' };
		}

		const minDate = dates[0];
		const maxDate = dates[dates.length - 1];
		const dateRange = this.formatDateRange(minDate, maxDate);

		return { minDate, maxDate, dateRange };
	}

	/**
	 * Deep clone an object
	 */
	static deepClone(obj) {
		if (obj === null || typeof obj !== 'object') return obj;
		if (obj instanceof Date) return new Date(obj.getTime());
		if (obj instanceof Array) return obj.map(item => this.deepClone(item));
		if (typeof obj === 'object') {
			const clonedObj = {};
			for (const key in obj) {
				if (obj.hasOwnProperty(key)) {
					clonedObj[key] = this.deepClone(obj[key]);
				}
			}
			return clonedObj;
		}
		return obj;
	}

	/**
	 * Merge objects deeply
	 */
	static deepMerge(target, source) {
		const result = this.deepClone(target);
		
		for (const key in source) {
			if (source.hasOwnProperty(key)) {
				if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
					result[key] = this.deepMerge(result[key] || {}, source[key]);
				} else {
					result[key] = source[key];
				}
			}
		}
		
		return result;
	}

	/**
	 * Debounce function execution
	 */
	static debounce(func, wait, immediate = false) {
		let timeout;
		return function executedFunction(...args) {
			const later = () => {
				timeout = null;
				if (!immediate) func.apply(this, args);
			};
			const callNow = immediate && !timeout;
			clearTimeout(timeout);
			timeout = setTimeout(later, wait);
			if (callNow) func.apply(this, args);
		};
	}

	/**
	 * Throttle function execution
	 */
	static throttle(func, limit) {
		let inThrottle;
		return function() {
			const args = arguments;
			const context = this;
			if (!inThrottle) {
				func.apply(context, args);
				inThrottle = true;
				setTimeout(() => inThrottle = false, limit);
			}
		};
	}
}

/**
 * Data validation utilities
 */
class ValidationUtils {
	/**
	 * Validate email format
	 */
	static isValidEmail(email) {
		const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
		return emailRegex.test(email);
	}

	/**
	 * Validate required fields
	 */
	static validateRequired(obj, requiredFields) {
		const errors = [];
		for (const field of requiredFields) {
			if (!obj[field] || (typeof obj[field] === 'string' && obj[field].trim() === '')) {
				errors.push(`${field} is required`);
			}
		}
		return errors;
	}

	/**
	 * Validate numeric range
	 */
	static validateRange(value, min, max) {
		const num = parseFloat(value);
		return !isNaN(num) && num >= min && num <= max;
	}

	/**
	 * Validate date format
	 */
	static isValidDate(dateString) {
		const date = new Date(dateString);
		return date instanceof Date && !isNaN(date);
	}
}

/**
 * Array and collection utilities
 */
class ArrayUtils {
	/**
	 * Group array by key
	 */
	static groupBy(array, key) {
		return array.reduce((groups, item) => {
			const group = item[key];
			groups[group] = groups[group] || [];
			groups[group].push(item);
			return groups;
		}, {});
	}

	/**
	 * Sort array by multiple keys
	 */
	static sortByMultiple(array, keys) {
		return array.sort((a, b) => {
			for (const key of keys) {
				const aVal = a[key];
				const bVal = b[key];
				if (aVal < bVal) return -1;
				if (aVal > bVal) return 1;
			}
			return 0;
		});
	}

	/**
	 * Remove duplicates from array
	 */
	static unique(array, key = null) {
		if (key) {
			const seen = new Set();
			return array.filter(item => {
				const value = item[key];
				if (seen.has(value)) {
					return false;
				}
				seen.add(value);
				return true;
			});
		}
		return [...new Set(array)];
	}

	/**
	 * Chunk array into smaller arrays
	 */
	static chunk(array, size) {
		const chunks = [];
		for (let i = 0; i < array.length; i += size) {
			chunks.push(array.slice(i, i + size));
		}
		return chunks;
	}
}

/**
 * String utilities
 */
class StringUtils {
	/**
	 * Capitalize first letter
	 */
	static capitalize(str) {
		return str.charAt(0).toUpperCase() + str.slice(1);
	}

	/**
	 * Convert to camel case
	 */
	static toCamelCase(str) {
		return str.replace(/-([a-z])/g, (g) => g[1].toUpperCase());
	}

	/**
	 * Convert to kebab case
	 */
	static toKebabCase(str) {
		return str.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
	}

	/**
	 * Truncate string with ellipsis
	 */
	static truncate(str, length, suffix = '...') {
		if (str.length <= length) return str;
		return str.substring(0, length - suffix.length) + suffix;
	}

	/**
	 * Escape HTML characters
	 */
	static escapeHtml(str) {
		const div = document.createElement('div');
		div.textContent = str;
		return div.innerHTML;
	}
}

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
	module.exports = {
		QUESTION_TYPES,
		NON_SCOREABLE_TYPES,
		ULTRA_QUESTION_TYPES,
		CoreUtils,
		ValidationUtils,
		ArrayUtils,
		StringUtils
	};
} 